<?php
/**
 * Plugin Name: Rapid Category Navigator
 * Plugin URI: https://libreproducts.info/rcn/
 * Description: 子カテゴリや孫カテゴリをAJAXで入れ子にして取得するプラグイン
 * Version: 2.8
 * Author: Near
 * Author URI: https://libreproducts.info/
 * License: GPL2
 * Text Domain: category-ajax-fetch
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// 各種ファイルをインクルード
require_once plugin_dir_path(__FILE__) . 'includes/admin-functions.php';
require_once plugin_dir_path(__FILE__) . 'includes/ajax-functions.php';


/**
 * フロントエンド用アセット（CSS/JS）の登録
 *
 * has_shortcode($post->post_content, ...) だけに依存すると、
 * 再利用ブロック/テンプレートパーツ/ウィジェット/フィルター挿入等で
 * post_content にショートコードが存在しないケースで読み込みが行われず、
 * 「何も表示されない」原因になります。
 *
 * そのため、ショートコード実行時にも必ず enqueue できるようにしておきます。
 */
function caf_enqueue_assets() {
    // CSS
    wp_enqueue_style(
        'category-ajax-fetch-styles',
        plugin_dir_url(__FILE__) . 'css/style.css',
        array(),
        '2.7'
    );

    // JS
    wp_enqueue_script(
        'category-ajax-fetch-js',
        plugin_dir_url(__FILE__) . 'js/category-ajax-fetch.js',
        array(),
        '2.7',
        true
    );

    // Ajax URL をJSに渡す
    wp_localize_script(
        'category-ajax-fetch-js',
        'categoryAjax',
        array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('caf_fetch_nonce'),
        )
    );
}


/**
 * フロントエンド用スクリプトの登録（ショートコードがあるページだけ読み込む）
 */
function enqueue_category_ajax_fetch_script() {

    if (is_admin()) {
        return;
    }

    $post = get_post();
    if (!$post) {
        return;
    }

    // 本文にショートコードが無ければ読み込まない
    if (!has_shortcode($post->post_content, 'rcn_view')) {
        return;
    }

    // ショートコード検出できたページは通常通り読み込み（保険）
    caf_enqueue_assets();
}

add_action('wp_enqueue_scripts', 'enqueue_category_ajax_fetch_script');


/**
 * ショートコードのコールバック関数
 * [rcn_view]
 */
function wp_column_directory_anchor_shortcode($atts) {

    // ★ショートコードが実行された時点で必ずアセットを読み込む
    // post_content に存在しないケース（再利用ブロック等）でも確実に動作させるため。
    caf_enqueue_assets();

    $atts = shortcode_atts(
        array(
            'id' => 'wp-column-directory-anchor', // デフォルトのID
        ),
        $atts,
        'rcn_view'
    );

    return '<div id="' . esc_attr($atts['id']) . '" data-caf-anchor="1"></div>';
}
add_shortcode('rcn_view', 'wp_column_directory_anchor_shortcode');


/**
 * プラグイン一覧に設定リンクを追加
 */
function caf_add_settings_link($links) {
    $settings_link = '<a href="tools.php?page=caf_settings">設定</a>';
    array_unshift($links, $settings_link);
    return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'caf_add_settings_link');